// JavaScript. Specifically oolite.jsVersion 185, an Oolite specific variant of ECMAv5.
// N.B. Use "const" and "let" where appropriate, but avoid all other ES6 features.
// N.B. Do not use template literals to compose message texts. 

"use strict";

// worldScripts.WitchBank

this.name = "WitchBank";

this._firstRun = function () {
    if (player.ship.awardEquipment("EQ_WB_WALLET")) {             // Dummy equipment to represent wallet
        missionVariables.WitchBank_cashAtBank = 0;                // Default starting value
        missionVariables.WitchBank_cashInWallet = player.credits; // Initial wallet balance equals total credits
    } else {
        log(this.name, "Error: EQ_WB_WALLET could not be awarded to player ship.")
    }
};

this.startUp = function () {
    if (!missionVariables.WitchBank_cashAtBank || 
        !missionVariables.WitchBank_cashInWallet) {
        this._firstRun()
    }
    delete this.startUp;
};

/* ====================================================================================
		equipmentDamaged & equipmentRepaired for EQ_WB_WALLET

    The EQ_WB_WALLET equipment doesn't do anything, except exist.
    Damage probability and cost-to-repair can be set in equipment.plist
    and handled natively by Oolite.
======================================================================================= */

// The equipmentDamaged handler is called when equipment gets damaged.
this.equipmentDamaged = function (equipment) {
    if (equipment === "EQ_WB_WALLET") {
        player.credits = missionVariables.WitchBank_cashAtBank;
        missionVariables.WitchBank_trappedInDamagedWallet = missionVariables.WitchBank_cashInWallet;
        missionVariables.WitchBank_cashInWallet = 0;
        player.consoleMessage("Your wallet has been damaged. " + missionVariables.WitchBank_trappedInDamagedWallet + " credits are inaccessible.", 6);
        log(this.name, "Wallet damaged. Trapped credits: " + missionVariables.WitchBank_trappedInDamagedWallet);
    }
};

// The equipmentRepaired handler is called when equipment gets repaired.
this.equipmentRepaired = function (equipment) {
    if (equipment === "EQ_WB_WALLET") {
        if (missionVariables.WitchBank_trappedInDamagedWallet > 0) {
            missionVariables.WitchBank_cashInWallet = missionVariables.WitchBank_trappedInDamagedWallet;
            player.credits = missionVariables.WitchBank_cashAtBank + missionVariables.WitchBank_cashInWallet;
            player.consoleMessage("Your wallet has been repaired. " + missionVariables.WitchBank_cashInWallet + " credits are restored.", 6);
            missionVariables.WitchBank_trappedInDamagedWallet = 0;
            log(this.name, "Wallet repaired. Restored credits: " + missionVariables.WitchBank_cashInWallet);
        } else {
            player.consoleMessage("No trapped credits to restore.", 6);
        }
    }
};

/* ====================================================================================
			SETTING THE STATION INTERFACE
======================================================================================= */

    this.startUpComplete = function () {
        if (player.ship.docked) {
            this.shipDockedWithStation(player.ship.dockedStation);
        }
        delete this.startUpComplete;
    };

    this.shipDockedWithStation = function (station) {
        if (station.isMainStation) {
            station.setInterface("witch_bank", {
            title: "Witch Bank",
            category: expandDescription("[interfaces-category-organisations]"),
            summary: "Witch Bank is the trusted name in interstellar banking. Current balance: " + missionVariables.WitchBank_cashAtBank + " credits.",
            callback: this._showBankingScreen.bind(this)
            });
        }
    };

/* ====================================================================================
            SHOW BANKING SCREEN
======================================================================================= */

this._pendingTransaction = null; // "deposit" or "withdrawal" - anything else is ignored

this._showBankingScreen = function (caller) {
    // If _pendingTransaction is set, process the caller as an amount
    if (this._pendingTransaction) {
        // Check if caller is a positive integer
        var amount = parseInt(caller, 10);
        if (!isNaN(amount) && amount > 0) {
            // Process the transaction
            this._performTransaction(this._pendingTransaction, amount);

            // Reset the pending transaction flag
            this._pendingTransaction = null;
        } else {
            player.consoleMessage("Invalid amount entered.", 6);
            this._pendingTransaction = null; // Reset on invalid input
        }
    }

    // Compose the default starting screen message
    var message = "Welcome to Witch Bank, the trusted name in interstellar banking. How can we help you today? Your current balances are: " +
                  missionVariables.WitchBank_cashAtBank + " credits in bank, " +
                  missionVariables.WitchBank_cashInWallet + " credits in wallet.";

    var optionsMenu = {
        "1_DEPOSIT": "Make a DEPOSIT",
        "2_WITHDRAWAL": "Make a WITHDRAWAL",
        "3_blank": "",
        "4_EXIT": "EXIT secure banking"
    };

    var parameters = {
        screenID: "BANKING_SCREEN",
        allowInterrupt: true,
        exitScreen: "GUI_SCREEN_INTERFACES",
        background: { name: "witchbank_background.png", height: 480 },
        title: "Witch Bank",
        message: message,
        choices: optionsMenu
    };

    // Adjust parameters for deposit/withdrawal screens
    switch (caller) {
        case "deposit":
            this._pendingTransaction = "deposit";
            parameters.title = "Wallet to Witch Bank";
            parameters.message = "Enter the amount you wish to deposit:";
            parameters.textEntry = true; // Replaces "choices" with a text entry field
            break;

        case "withdrawal":
            this._pendingTransaction = "withdrawal";
            parameters.title = "Witch Bank to Wallet";
            parameters.message = "Enter the amount you wish to withdraw:";
            parameters.textEntry = true; // Replaces "choices" with a text entry field
            break;

        default:
            // No changes to the starting screen
            break;
    }

    // Run the banking screen
    mission.runScreen(parameters, this._handleBankingChoice.bind(this));
};

/* ====================================================================================
            HANDLE BANKING CHOICE
======================================================================================= */

this._handleBankingChoice = function (choice) {
    if (choice === "4_EXIT") {
        return; // Exit the banking screen
    }

    if (choice === "1_DEPOSIT") {
        this._showBankingScreen("deposit");
    } else if (choice === "2_WITHDRAWAL") {
        this._showBankingScreen("withdrawal");
    } else {
        // Update text on F4 screen with new balance
        player.ship.dockedStation.setInterface("witch_bank", {
            title: "Witch Bank",
            category: expandDescription("[interfaces-category-organisations]"),
            summary: "Witch Bank is the trusted name in interstellar banking. Current balance: " + missionVariables.WitchBank_cashAtBank + " credits.",
            callback: this._showBankingScreen.bind(this)
        });
        // Redisplay the banking screen
        this._showBankingScreen("witch_bank");
    }
};

/* ====================================================================================
            PERFORM TRANSACTION
======================================================================================= */

this._performTransaction = function (transType, amount) {
    switch (transType) {
        case "deposit":
            // Transfer credits from wallet to bank
            if (amount > 0 && amount <= missionVariables.WitchBank_cashInWallet) {
                missionVariables.WitchBank_cashAtBank += amount;
                missionVariables.WitchBank_cashInWallet -= amount;
                player.consoleMessage("Deposited " + amount + " credits.", 6);
                log(this.name, "Deposited " + amount + " credits. New balances: Bank=" +
                    missionVariables.WitchBank_cashAtBank + ", Wallet=" +
                    missionVariables.WitchBank_cashInWallet);
            } else {
                player.consoleMessage("Invalid deposit amount.", 6);
            }
            break;

        case "withdrawal":
            // Transfer credits from bank to wallet
            if (amount > 0 && amount <= missionVariables.WitchBank_cashAtBank) {
                missionVariables.WitchBank_cashAtBank -= amount;
                missionVariables.WitchBank_cashInWallet += amount;
                player.consoleMessage("Withdrew " + amount + " credits.", 6);
                log(this.name, "Withdrew " + amount + " credits. New balances: Bank=" +
                    missionVariables.WitchBank_cashAtBank + ", Wallet=" +
                    missionVariables.WitchBank_cashInWallet);
            } else {
                player.consoleMessage("Invalid withdrawal amount.", 6);
            }
            break;

        default:
            player.consoleMessage("Invalid banking operation.", 6);
            break;
    }
};

/* ====================================================================================
            COMMON TRANSACTIONS
======================================================================================= */

/* ====================================
    THESE WILL BE USEFUL LATER

playerBoughtCargo

The playerBoughtCargo handler is called when cargo is bought at the market screen. commodity contains the non-localised name for the cargo. You can get the localised name using expandDescription("[commodity-name "+commodity+"]");. price is price per unit in tenths of a credit.

this.playerBoughtCargo = function(commodity, units, price)
{
     // Your code here
}

playerSoldCargo

The playerSoldCargo handler is called when cargo is sold at the market screen. commodity contains the non-localised name for the cargo. You can get the localised name using expandDescription("[commodity-name "+commodity+"]");. price is price per unit in tenths of a credit.

this.playerSoldCargo = function(commodity, units, price)
{
     // Your code here
}

playerBoughtEquipment

The playerBoughtEquipment handler is called when equipment is bought at the outfit screen.

this.playerBoughtEquipment = function(equipment, paid)
{
     // Your code here
}
The 'paid' parameter is the amount of credits the player paid for the equipment, including any refunds that might have been applied during the purchase (eg when purchasing a laser and installing it in a position where a laser is already fitted, the cost of the current laser will be refunded to the player during the purchase).

playerBoughtNewShip

The playerBoughtNewShip handler is called when a new ship is bought.

this.playerBoughtNewShip = function(ship, price)
{
     // Your code here
}
The 'price' parameter is the cost of the ship (not counting any trade-in value of the player's old ship) in credits, or zero for changes using player.replaceShip()
======================================= */


/* ====================================================================================
            FANCY FUTURE FEATURES

5. Risk Simulation
The _checkForBankFailure function introduces a 1% chance of bank failure per visit, wiping out the player's bank balance. This adds tension but might feel punishing if it happens frequently.
Suggestion: Mitigate the risk by introducing an insurance system or warning messages:
javascript
this._checkForBankFailure = function () {
    if (Math.random() < 0.01) { // 1% chance per visit
        missionVariables.WitchBank_cashAtBank = 0;
        player.consoleMessage("WARNING: Your bank has failed! All deposits lost.", 10);
        log(this.name, "Bank failure occurred. Player lost all cash-at-bank.");
    } else if (Math.random() < 0.05) { // 5% chance of instability warning
        player.consoleMessage("WARNING: Your bank appears unstable. Consider withdrawing funds.", 6);
    }
};

Future Enhancements
Interest on Deposits: Introduce a small interest rate to incentivize saving.
Insurance System: Allow players to purchase insurance to protect against bank failures or wallet damage.
Loan System: Let players take out loans with interest rates, adding another layer of financial management.
4. Wallet Capacity
Introduce a maximum limit for cashInWallet to encourage players to deposit excess credits:
javascript
var maxWalletCapacity = 100000; // Example: Maximum 100,000 credits in wallet
if (missionVariables.WitchBank_cashInWallet > maxWalletCapacity) {
    player.consoleMessage("Your wallet is full. Deposit excess credits in the bank.", 6);
}

6. Integration with Other Events
You’ve included placeholders for playerBoughtCargo, playerSoldCargo, playerBoughtEquipment, and playerBoughtNewShip. These can be used to simulate real-world banking scenarios:
Deduct transaction fees when the player buys or sells cargo.
Offer loans or credit lines when purchasing expensive equipment or ships.
Example for playerBoughtCargo:
javascript
this.playerBoughtCargo = function (commodity, units, price) {
    var fee = Math.ceil((price * units) / 100); // 1% transaction fee
    missionVariables.WitchBank_cashAtBank -= fee;
    player.consoleMessage("Transaction fee of " + fee + " credits deducted from your bank account.", 6);
};


======================================================================================= */

this._checkForBankFailure = function () {
    if (Math.random() < 0.01) { // 1% chance per visit
        missionVariables.WitchBank_cashAtBank = 0;
        player.consoleMessage("Your bank has failed! All deposits lost.", 10);
    }
};

this._applyInterest = function () {
    var interestRate = 0.01; // 1% per day
    var interest = Math.floor(missionVariables.WitchBank_cashAtBank * interestRate);
    missionVariables.WitchBank_cashAtBank += interest;
    player.consoleMessage("You earned " + interest + " credits in interest.", 6);
};


/* ====================================================================================

Feedback and Suggestions
1. Initialization Logic
The _firstRun function initializes the banking system by awarding the EQ_WB_WALLET equipment and setting default values for cashAtBank and cashInWallet.
Suggestion: Add a fallback mechanism in case missionVariables properties are undefined or corrupted during gameplay:
javascript
if (!missionVariables.WitchBank_cashAtBank) {
    missionVariables.WitchBank_cashAtBank = 0;
}
if (!missionVariables.WitchBank_cashInWallet) {
    missionVariables.WitchBank_cashInWallet = player.credits;
}

======================================================================================= */







/* ====================================================================================
			BANKING SCREENS


    // WRONG: Fallback message if messageKey is not in localization file
    // RIGHT: message over-rides messageKey

Keys needed to include in missiontext.plist file (so far).

    "banking_title" = "Witch Bank";
    "banking_message" = "Welcome to Witch Bank, the trusted name in interstellar banking. How can we help you today? Your current balance is: ";
    "banking_choices" = "";

    "banking_deposit_title" = "Wallet to Bank";
    "banking_deposit_message" = "";
    "banking_deposit_choices" = "";

    "banking_withdrawal_title" = "Bank to Wallet";
    "banking_withdrawal_message" = "";
    "banking_withdrawal_choices" = "";

======================================================================================= */

this._pendingTransaction = null; // "deposit" or "withdrawal" - anything else is to be ignored

this._showBankingScreen = function (caller) {
    // If the _pendingTransaction flag is set, we have to:

    // TODO: 1. catch any values of caller that look like numbers

    // 2. Process the transaction, either deposit or withdrawal
    this._performTransaction = function (this._pendingTransaction, amount)

    // 3. Show a help message if the number entered is out of range
    // 4. Reset _pendingTransaction to null
    this._pendingTransaction = null;

    // 5. Continue on to the default starting screen message, which
    //    will be composed using the updated balances.
     
    // Parameters for the starting screen

    var message = "Welcome to Witch Bank. Your current balance: " +
                  missionVariables.WitchBank_cashAtBank + " credits in bank, " +
                  missionVariables.WitchBank_cashInWallet + " credits in wallet.";

    var optionsMenu = {
        "1_DEPOSIT" = "Make a DEPOSIT",
        "2_WITHDRAWAL" = "Make a WITHDRAWAL",
        "3_blank" = "",
        "4_EXIT" = "EXIT secure banking"
    };
    var parameters = {
        screenID: "BANKING_SCREEN",
        allowInterrupt: true,
        exitScreen: "GUI_SCREEN_INTERFACES",
        background: { name: "witchbank_background.png", height: 480 },
     // titleKey:   "banking_title",
        title: "Witch Bank",
     // messageKey: "banking_message",
        message: message,
        choices: optionsMenu
    };
    // Adjust parameters for displayed text, if not the starting screen
    switch (caller) {
         case "deposit":
            this._pendingTransaction = "deposit";
            parameters.titleKey  = "banking_deposit_title";
            parameters.messageKey = "banking_deposit_message";
            parameters.textEntry = true; // replaces "choices" with text entry field.
            break;

        case "withdrawal":
            this._pendingTransaction = "withdrawal";
            parameters.titleKey   = "banking_withdrawal_title";
            parameters.messageKey = "banking_withdrawal_message";
            parameters.textEntry = true; // replaces "choices" with text entry field.
            break;

        default:
            // No changes to starting screen
            break;
    }

    mission.runScreen(parameters, callback);

        function callback(choice) {
            if (choice !== "4_EXIT") {
                if (choice === "1_DEPOSIT") {
                    this._showBankingScreen("deposit");
                } else if (choice === "2_WITHDRAWAL") {
                    this._showBankingScreen("withdrawal");
                } else {
                    this._applyInterface(player.ship.dockedStation.setInterface("witch_bank", {
                    title: "Witch Bank",
                    category: expandDescription("[interfaces-category-organisations]"),
                    summary: "Witch Bank is the trusted name in interstellar banking. Current balance: " + missionVariables.WitchBank_cashAtBank + " credits.",
                    callback: this._showBankingScreen.bind(this)
                    });;
                    this._showBankingScreen("witch_bank");
                }
            }
        }
    
    }
};












